/* Adobe.Particles.js */

define(["lib/Zoot", 	"src/math/Random",
	"lib/dev", "lib/tasks", "lib/tasks/treeOfMatrix"],
function (Z,				Random,
	dev, tasks, treeOfMatrix) {
	"use strict";

	var kAbout = "$$$/private/animal/Behavior/Particles/About=Particles, (c) 2014.",
			vec2 = Z.Vec2,
			m3 = Z.Mat3,
			rKeyCode = Z.keyCodes.getKeyGraphId("R"),
			dKeyCode = Z.keyCodes.getKeyGraphId("D");

	var kNoAuto = {
		translation : false,
		linear : false
	};

	var kAffine = tasks.dofs.type.kAffine;

	return {
		about: kAbout,
		description: "$$$/animal/Behavior/Particles/Desc=Replicates layers to create many moving particles; good for snow/rain etc.",
		uiName: "$$$/animal/Behavior/Particles/UIName=Particles",
		defaultArmedForRecordOn: true,

		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			return [

				{ id: "ParticleMode", type: "enum", uiName: "$$$/animal/Behavior/Particles/Parameter/ParticleMode=Particle Mode",
					items: [{ id: 0, uiName: "$$$/animal/Behavior/Particles/Parameter/ParticleMode/Snow=Snow" },
							{ id: 1, uiName: "$$$/animal/Behavior/Particles/Parameter/ParticleMode/PointAndShoot=Point and Shoot" },
							{ id: 2, uiName: "$$$/animal/Behavior/Particles/Parameter/ParticleMode/Cannon=Cannon"}],
							uiToolTip: "$$$/animal/Behavior/Particles/Parameter/ParticleMode/tooltip=Select particle creation method", dephault: 1
				},
				{ id: "MouseInput", type: "eventGraph", uiName: "$$$/animal/Behavior/Particles/Parameter/MouseInput=Mouse Input", inputKeysArray: ["Mouse/"],
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/MouseInput/tooltip=Mouse input used to interact with particle creation and movement", defaultArmedForRecordOn: true },
				{ id: "KeyboardInput", type: "eventGraph", uiName: "$$$/animal/Behavior/Particles/Parameter/KeyboardInput=Keyboard Input", inputKeysArray: [rKeyCode, dKeyCode],
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/KeyboardInput/tooltip=Keyboard input used to interact with particles", defaultArmedForRecordOn: true },
				// {id:"Smoke", type: "checkbox", uiName: "$$$/animal/Behavior/Particles/Parameter/Smoke=Smoke",dephault: false},
				{id: "ContinuousMode", type: "checkbox", uiName: "$$$/animal/Behavior/Particles/Parameter/ContinuousMode=Continuous Mode",
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/ContinuousMode/tooltip=Continuously emit particles when mouse pointer is over the Scene panel", dephault: false },
				{ id: "ParticlesPerSecond", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/ParticlesPerSecond=Particles Per Second",
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/ParticlesPerSecond/tooltip=Set rate at which particles are created", min: 0, precision: 1, dephault: 10 },
				{ id: "Direction", type: "angle", uiName: "$$$/animal/Behavior/Particles/Parameter/Direction=Direction", precision: 0,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Direction/tooltip=Set starting direction of Snow and Cannon particles", dephault: 180 },
				{ id: "Velocity", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/Velocity=Velocity", uiUnits: "px/sec", min: 0, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Velocity/tooltip=Set starting speed of particles", dephault: 100 },
				{ id: "Spread", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/Spread=Spread", uiUnits: "%", min: 0, max: 100, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Spread/tooltip=Widen the stream of created particles around the starting direction", dephault: 0 },
				{ id: "Randomness", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/Randomness=Randomness", uiUnits: "%", min: 0, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Randomness/tooltip=Randomize starting velocity of a particle", dephault: 0 },
				{ id: "Lifespan", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/Lifespan=Lifespan", uiUnits: "$$$/animal/Behavior/Particles/units/sec=sec", min: 0, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Lifespan/tooltip=Set the lifetime (duration) of a particle", dephault: 5 },
				{ id: "GravityMagnitude", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/GravityMagnitude=Gravity Strength", uiUnits: "px/sec^2", min: 0, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/GravityMagnitude/tooltip=Adjust the pull strength of gravity on particles", dephault: 100 },
				{ id: "GravityDirection", type: "angle", uiName: "$$$/animal/Behavior/Particles/Parameter/GravityDirection=Gravity Direction", precision: 0,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/GravityDirection/tooltip=Set direction of gravity", dephault: 180 },
				{ id: "CollisionWithWalls", type: "checkbox", uiName: "$$$/animal/Behavior/Particles/Parameter/CollisionWithWalls=Bounce Off Scene Sides",
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/CollisionWithWalls/tooltip=Make particles bounce when hitting scene bounds", dephault: true },
				{ id: "Bounciness", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/Bounciness=Bounciness", uiUnits: "%", min: 0, max: 100, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Bounciness/tooltip=Adjust how much colliding particles bounce", dephault: 90 },
				{ id: "SquashAndStretch", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/SquashAndStretch=Squash and Stretch", uiUnits: "%", min: 0, max: 100, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/SquashAndStretch/tooltip=Make particles deformable", dephault: 0 },
				{ id: "Wind", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/Wind=Wind Strength", uiUnits: "%", min: 0, max: 100, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Wind/tooltip=Adjust impact of wind blowing on particles; click in Scene panel to direct the wind", dephault: 0 },
				{ id: "Drag", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/Drag=Drag", uiUnits: "%", min: 0, max: 100, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/Drag/tooltip=Decelerate particles as if affected by friction", dephault: 0 },
				{ id: "HonorEmitterRotationAndScale", type: "checkbox", uiName: "$$$/animal/Behavior/Particles/Parameter/HonorEmitterRotationAndScale=Honor Emitter Transform",
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/HonorEmitterRotationAndScale/tooltip=Allow particle to inherit scale, skew, and rotation of emitter puppet", dephault: true },
				{ id: "ParticleScale", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/ParticleScale=Particle Scale", uiUnits: "%", min: 1, max: 100, precision: 1,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/ParticleScale/tooltip=Set size of a particle when it is born", dephault: 100 },
				{ id: "EmitterAlpha", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/EmitterAlpha=Emitter Opacity", uiUnits: "%", min: 0, max: 100, precision: 0,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/EmitterAlpha/tooltip=Set transparency of emitter", dephault: 100 },
				{ id: "ParticleAlpha", type: "slider", uiName: "$$$/animal/Behavior/Particles/Parameter/ParticleAlpha=Particle Opacity", uiUnits: "%", min: 0, max: 100, precision: 0,
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/ParticleAlpha/tooltip=Set transparency of a particle when it is born", dephault: 100 },
				{ id: "LifespanAlpha", type: "checkbox", uiName: "$$$/animal/Behavior/Particles/Parameter/LifespanAlpha=Fade Particle Opacity",
					uiToolTip: "$$$/animal/Behavior/Particles/Parameter/LifespanAlpha/tooltip=Fade transparency of a particle over its lifespan", dephault: false }
			];
		},

		onCreateStageBehavior: function (self, args) {
			self.lastDown = 0;
			self.aParticleLayer = [];
			self.velocity = [];
			self.position = [];
			self.timer = [];
			self.lifetimeTime = [];
			self.particleScale = [];
			self.aMatLayer_Scene = [];

			self.particleTrack = new Z.Track("ParticleTrack", true);

			// For now, only warp-independent puppets can break off as a particle...
			self.sourceParticleLayer = args.stageLayer.privateLayer.getWarperLayer();

			// intialized in first onAnimate call
			self.sourceTrackItem = null; 
			self.scene = null;
			
			// initialize the rehearsal data.
			this.onResetRehearsalData(self);

			self.killAll = false;
		},

		onResetRehearsalData : function (self,args) {
			var seed = 123456789;
			self.rand = new Random(seed);
			self.last_t = undefined;
			self.stepperTimeResidual = 0.0;
			self.particleBirthResidual = 0;
			self.killAll = true;
		},
		
		// TODO: duplicated in Adobe.Blinker.js -- need to refactor!
		setSeedFromArgs : function (self, args) {
			var seed = args.t * 653 + args.globalRehearseTime * 109;
		
			// Mix in the behaviorInstanceId so we don't get repeated patterns for duplicated puppets.
			var idStr = args.stageLayer.getId();
			for (var i=0; i<idStr.length; ++i) {
				seed += idStr.charCodeAt(i);
			}
			self.rand.setSeed(seed);
		},

		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage

			var rDown = args.getParamEventValue("KeyboardInput", rKeyCode) || 0,
				dDown = args.getParamEventValue("KeyboardInput", dKeyCode) || 0,
				dof, invMtx, track,
				i, p,
				mousePosition = args.getParamEventValue("MouseInput", "Mouse/Position"), 
				overValue = args.getParamEventValue("MouseInput", "Mouse/Over"),
				leftDownB = args.getParamEventValue("MouseInput", "Mouse/Down/Left"),
				dt, lifetimeTime = args.getParam("Lifespan"),
				CoR = args.getParam("Bounciness") / 100.0,
				normalCoR = CoR,
				angularCoR = (CoR + 0.1) / 2.0,
				tangentAngularCoR = 0.01,
				gravityAngle = 2 * Math.PI / 360 * args.getParam("GravityDirection") - Math.PI / 2,
				gravity = [args.getParam("GravityMagnitude") * Math.cos(gravityAngle), args.getParam("GravityMagnitude") * Math.sin(gravityAngle)],
				view = self.sourceParticleLayer.getView(),
				width = view.getWidth(),
				height = view.getHeight(),
				topY = -height/2,
				bottomY = height/2,
				leftX = -width/2,
				rightX = width/2,
				v, pos,
				vx, vtheta, vscale, px, ptheta, pscale, pshear,
				normal, scaleRest,
				Q, c, s, particleScale, eps,
				scaleStiffness, scaleDamping,
				mDof, force, DeformationOn, spreadVector, particleMode;

			self.sourceParticleLayer.setOpacity(self.sourceParticleLayer.getOpacity() * args.getParam("EmitterAlpha")/100);

			if (args.getParam("ParticleMode") === 0) {
				particleMode = "Snow";
			} else if (args.getParam("ParticleMode") === 1) {
				particleMode = "PointAndShoot";
			} else {
				particleMode = "Cannon";
			}

			if (! self.sourceTrackItem) {
				self.sourceTrackItem = self.sourceParticleLayer.getTrackItem();
				self.scene = self.sourceParticleLayer.getScene();
				track = self.sourceParticleLayer.getTrack();
				self.particleTrack.setParentOffsetTime(track.getParentOffsetTime());
				self.particleTrack.setTrimInTime(track.getTrimInTime());
				self.particleTrack.setTrimOutTime(track.getTrimOutTime());
				var idx = track.getParent().getChildren().indexOf(track);
				self.scene.addChild(self.particleTrack, idx+1);
			}

			// Time book-keeping
			var curr_t = args.t + args.globalRehearseTime; 
			this.setSeedFromArgs(self, args);
			
			// console.logToUser("args.t = " + args.t);
			
			if (self.last_t === undefined) {
				self.last_t = curr_t;
			}
			var delta_t = curr_t - self.last_t;
			self.last_t = curr_t;

			// var delta_t = 1/args.scene_.getFrameRate();
			self.stepperTimeResidual = self.stepperTimeResidual + delta_t;
			
			var dt_default = 1.0 / 24.0;
			dt = dt_default;

			var pps = args.getParam("ParticlesPerSecond");
			if(pps > 0){
				self.particleBirthResidual = self.particleBirthResidual + Math.round(delta_t*pps*1000)/1000;
				dt = Math.min(dt_default,1.0/pps);
			}

			if( self.stepperTimeResidual < 0.0 || self.stepperTimeResidual > 1.1 ){ // 1.1 to prevent mass particle attack on scrub
				self.stepperTimeResidual = 0.0;
				self.particleBirthResidual = 0;
			}
			// console.logToUser("delta_t = " + delta_t);
			// console.logToUser("stepperTimeResidual = " + self.stepperTimeResidual);
			// console.logToUser("particleBirthResidual = " + self.particleBirthResidual);

			// Timestepper params
			var squashAndStretch = args.getParam("SquashAndStretch"),
				lifespanAlpha = args.getParam("LifespanAlpha"),
				particleAlpha = args.getParam("ParticleAlpha"),
				wind = args.getParam("Wind"),
				drag = args.getParam("Drag"),
				collisionWithWalls = args.getParam("CollisionWithWalls");

			// Timestepping
			while (self.stepperTimeResidual > dt) {
				self.stepperTimeResidual = self.stepperTimeResidual - dt;

				// Step particle dynamics
				if (squashAndStretch < 10.0) {
					DeformationOn = false;
					scaleStiffness = 100;
				} else {
					DeformationOn = true;
					scaleStiffness = 200.0 / squashAndStretch * 2.5;
				}
				scaleDamping = 2 * Math.sqrt(scaleStiffness);

				for (i = 0; i < self.aParticleLayer.length; i += 1) {

					p = self.aParticleLayer[i];
					if(lifespanAlpha){
						p.setOpacity(Math.min(1,Math.max(0,self.timer[i])/self.lifetimeTime[i] * particleAlpha/100));
					} else {
						p.setOpacity(particleAlpha/100);
					}
					dof = p.getHandleTreeRoot();

					// grab velocities
					// console.logToUser ("vel= " + self.velocity[i]);
					vx = [self.velocity[i][0], self.velocity[i][1]];
					vtheta = self.velocity[i][2];
					vscale = [self.velocity[i][3], self.velocity[i][4]];

					// grab positions`
					// console.logToUser ("pos= " + self.position[i]);
					px = [self.position[i][0], self.position[i][1]];
					ptheta = self.position[i][2];
					pscale = [self.position[i][3], self.position[i][4]];
					pshear = [self.position[i][5], self.position[i][6]];

					// for reference
					scaleRest = self.particleScale[i];

					// integrate forces:
					// 1. body forces
					force = [dt * gravity[0], dt * gravity[1]];
					vx = vec2.add(force, vx);

					// 2. wind forces
					if (wind > 0 && leftDownB) {
						args.setEventGraphParamRecordingValid("MouseInput");
						force = vec2.subtract(mousePosition, [-30, -30]);
						force = vec2.scale(dt * wind / 100, force);
						vx = vec2.add(force, vx);
					}

					// 3. "drag" forces
					if (drag) {
						force = vx;
						force = vec2.scale(-dt * drag / 100, force);
						vx = vec2.add(force, vx);
					}

					// 4. defo forces
					if (DeformationOn) {
						// scale dofs 
						if (Math.abs(pscale[0] - scaleRest[0]) > 0.01) {
							vscale[0] -= dt * scaleStiffness * (pscale[0] - scaleRest[0]) / Math.abs(pscale[0] - scaleRest[0]) + dt * (scaleDamping) * vscale[0];
						}
						if (Math.abs(pscale[1] - scaleRest[1]) > 0.01) {
							vscale[1] -= dt * scaleStiffness * (pscale[1] - scaleRest[1]) / Math.abs(pscale[1] - scaleRest[1]) + dt * (scaleDamping) * vscale[1];
						}

					} else {
						pscale = scaleRest;
						vscale = [0, 0];
					}

					// 5. contact forces
					if (collisionWithWalls) {

						// normal
						normal = [0, 0];

						// collision  processing 
						if (px[0] < leftX) {
							px[0] = leftX;
							vx[0] = -normalCoR * vx[0];
							vtheta = -angularCoR * vtheta;
							vtheta += tangentAngularCoR * vx[1];
							normal = [1, 0];

						} else if (px[0] > rightX) {
							px[0] = rightX;
							vx[0] = -normalCoR * vx[0];
							vtheta = -angularCoR * vtheta;
							vtheta += tangentAngularCoR * vx[1];
							normal = [-1, 0];
						}
						if (px[1] < topY && !(particleMode === "Snow" && vx[1] > 0)) {
							px[1] = topY;
							vx[1] = -normalCoR * vx[1];
							vtheta = -angularCoR * vtheta;
							vtheta += tangentAngularCoR * vx[0];
							normal = [0, 1];

						} else if (px[1] > bottomY) {
							px[1] = bottomY;
							vx[1] = -normalCoR * vx[1];
							vtheta = -angularCoR * vtheta;
							vtheta += tangentAngularCoR * vx[0];
							normal = [0, 1];
						}

						if (DeformationOn && Math.abs(normal[0]) + Math.abs(normal[1]) > 0) {
							Q = m3.rotation(ptheta);
							invMtx = m3.transpose(Q);
							vec2.transformAffine(invMtx, normal, normal);
							vscale[0] -= (1.0 / (20 * scaleStiffness)) * normal[0] * vx[0];
							vscale[1] -= (1.0 / (20 * scaleStiffness)) * normal[1] * vx[1];
						}
					}

					// integrate position
					px = vec2.axpy(dt, vx, px);
					ptheta += dt * vtheta;
					pscale[0] = Math.max(0.25 * scaleRest[0], pscale[0] + dt * vscale[0]);
					pscale[1] = Math.max(0.25 * scaleRest[1], pscale[1] + dt * vscale[1]);

					// store local state (position and velocity)
					self.velocity[i] = [vx[0], vx[1], vtheta, vscale[0], vscale[1]];
					self.position[i] = [px[0], px[1], ptheta, pscale[0], pscale[1], pshear[0], pshear[1]];
				}

				if(Math.floor(self.particleBirthResidual) > 0){
					self.particleBirthResidual = self.particleBirthResidual - 1;
			
					// Birthing
					if ( lifetimeTime > 0 && ((args.getParam("ContinuousMode") && particleMode === "Cannon") || (overValue && mousePosition) || particleMode === "Snow")) {
						var rChanged = rDown > self.lastDown, mouseBirth = leftDownB && !args.getParam("ContinuousMode");
						if (mouseBirth || args.getParam("ContinuousMode") || rChanged || particleMode === "Snow") {
							if (mouseBirth) {
								args.setEventGraphParamRecordingValid("MouseInput");
							}
							if (rChanged) {
								args.setEventGraphParamRecordingValid("KeyboardInput");
							}
							self.lastDown = rDown;

							// create new particle and corresponding track item
							var particleTrackItem = self.sourceTrackItem.clone(),
								particleLayer = self.sourceParticleLayer.clone();
							particleLayer.prepareState();

							particleTrackItem.setSourceLayer(particleLayer);
							self.particleTrack.addChild(particleTrackItem);
							particleTrackItem.displayInView(view);

							dof = particleLayer.getHandleTreeRoot();
							particleScale = args.getParam("ParticleScale") / 100.0;

							var matScene_LayerSource = args.getLayerMatrixRelativeToScene(self.sourceParticleLayer),
								matLayerSource_Handle, matScene_HandleSource,
								matScene_Layer = args.getLayerMatrixRelativeToScene(particleLayer),
								matLayer_Scene = m3.invert(matScene_Layer);

							matLayerSource_Handle = tasks.handle.getFrameRelativeToLayer(self.sourceParticleLayer.getHandleTreeRoot());
							matScene_HandleSource = m3.multiply(matScene_LayerSource, matLayerSource_Handle);

							mDof = m3(matScene_HandleSource);
							var rootPosition = [], rootScale = [], rootShear = [], rootRotation = [], rootAng, fireAngle;
							m3.decomposeAffine(mDof, rootPosition, rootScale, rootShear, rootRotation);
							rootAng = Math.atan2(rootRotation[3], rootRotation[2]) - Math.PI / 2;
							fireAngle = 2 * Math.PI / 360 * args.getParam("Direction") - Math.PI / 2;
							if(args.getParam("HonorEmitterRotationAndScale")){
								fireAngle += rootAng;
							}

							if (particleMode === "Snow") {
								c = Math.cos(fireAngle);
								s = Math.sin(fireAngle);
								v = [c, s];
								v = vec2.scale(args.getParam("Velocity"), v);
								px = [(args.getParam("Spread") / 100) * width / 2 * (1 - 2 * self.rand.random()), topY - 20.0];
							} else if (particleMode === "PointAndShoot") {
								v = vec2.subtract(mousePosition, rootPosition);
								v = vec2.normalize(v);
								spreadVector = [-v[1], v[0]];
								spreadVector = vec2.scale((args.getParam("Spread") / 100) * width / 10 * (1 - 2 * self.rand.random()), spreadVector);
								rootPosition = vec2.add(rootPosition, spreadVector);
								v = vec2.scale(args.getParam("Velocity"), v);
								px = [rootPosition[0], rootPosition[1]];
							} else if (particleMode === "Cannon") {
								c = Math.cos(fireAngle);
								s = Math.sin(fireAngle);
								v = [c, s];
								spreadVector = [-v[1], v[0]];
								spreadVector = vec2.scale((args.getParam("Spread") / 100) * width / 10 * (1 - 2 * self.rand.random()), spreadVector);
								rootPosition = vec2.add(rootPosition, spreadVector);
								v = vec2.scale(args.getParam("Velocity"), v);
								px = [rootPosition[0], rootPosition[1]];
							}

							// perturb initial velocity
							eps = args.getParam("Randomness") / 2;
							v = vec2.add([eps * (2 * self.rand.random() - 1), eps * (2 * self.rand.random() - 1)], v);

							// set total vel config
							v.push(0, 0, 0); // add angular and scale velocities

							// set total pos config
							if(args.getParam("HonorEmitterRotationAndScale")){
								mDof = m3.affine(px,[particleScale*rootScale[0], particleScale*rootScale[0]], rootShear, rootAng);
								pos = [px[0], px[1], rootAng, particleScale*rootScale[0], particleScale*rootScale[1], rootShear[0], /* y-rootShear= */ 0 ];
							} else {
								// just use pos of emitter
								mDof = m3.affine(rootPosition, [particleScale, particleScale], [ 0 ], 0);
								pos = [px[0], px[1], 0, particleScale, particleScale, 0, 0];
							}

							tasks.handle.setFrame(dof, tasks.handle.convertLayerFrame(dof, m3.multiply(matLayer_Scene, mDof)), kAffine);

							self.aParticleLayer.push(particleLayer);
							self.velocity.push(v);
							self.position.push(pos);
							self.timer.push(lifetimeTime);
							self.lifetimeTime.push(lifetimeTime);
							self.particleScale.push([pos[3], pos[4]]);
							self.aMatLayer_Scene.push(matLayer_Scene);
							// console.log("Created particle (" + self.aParticleLayer.length + " total)");
						}
					}
				}
			}

			// After stepping set final state
			for (i = 0; i < self.aParticleLayer.length; i += 1) {

				p = self.aParticleLayer[i];
				px = [self.position[i][0], self.position[i][1]];
				ptheta = self.position[i][2];
				pscale = [self.position[i][3], self.position[i][4]];
				pshear = [self.position[i][5], self.position[i][6]];

				// TODO: don't use on y-shear or compose such matrices without mat3.affine()
				mDof = m3.affine(px, pscale, [ pshear[0] ], ptheta);
				var matLayer_Dof = m3.multiply(self.aMatLayer_Scene[i], mDof);

				dof = p.getHandleTreeRoot();
				tasks.handle.setFrame(dof, tasks.handle.convertLayerFrame(dof, matLayer_Dof), kAffine);
			}

			// update lifespan left
			for (i = 0; i < self.aParticleLayer.length; i += 1) {
				self.timer[i] -= delta_t; //decrement timer
			}

			function removePuppetFromView(particleLayer)
			{
				var trackItem = particleLayer.getTrackItem(),
					view = particleLayer.getView();
				trackItem.removeFromView(view);
				self.particleTrack.removeChild(trackItem);
			}

			// Death:	
			if (dDown || self.killAll) {
				if (dDown) {
					args.setEventGraphParamRecordingValid("KeyboardInput");
				}
				if (self.killAll) {
					self.killAll = false;
				}
				for (i = 0; i < self.aParticleLayer.length; i += 1) {
					p = self.aParticleLayer[i];
					removePuppetFromView(p);
				}
				self.aParticleLayer = [];
				self.velocity = [];
				self.position = [];
				self.timer = [];
				self.lifetimeTime = [];
				self.particleScale = [];
				self.particleBirthResidual = 0;
				self.stepperTimeResidual = 0.0;
			}

			// check all for death
			for (i = 0; i < self.aParticleLayer.length; i += 1) {
				// console.log ("Particle " + i, " life left : " + (self.timer[i]) );
				if (self.timer[i] <= 0) {
					p = self.aParticleLayer[i];
					removePuppetFromView(p);
					self.aParticleLayer.splice(i,1);
					self.velocity.splice(i,1);
					self.position.splice(i,1);
					self.timer.splice(i,1);
					self.lifetimeTime.splice(i,1);
					self.particleScale.splice(i,1);
					console.log ("Killed particle " + i, " now " + self.aParticleLayer.length + " particles left");
				}
			}
		}

	}; // end of object being returned
});
